<?php

declare(strict_types=1);

namespace HHIT\ConfigGenerator\Generator\Values;

use HHIT\ConfigGenerator\Generator\IO\File;
use HHIT\ConfigGenerator\Generator\IO\JsonReader;
use HHIT\ConfigGenerator\Generator\Secrets\SecretProviderInterface;

class ValuesLoader
{
    private JsonReader $valuesReader;
    private ?JsonReader $valuesEnvReader;
    private array $context = [];
    private SecretProviderInterface $provider;
    private string $env;

    public function __construct(
        File                    $valuesFile,
        ?File                   $valuesEnvFile,
        SecretProviderInterface $provider,
        string                  $env
    ) {
        $this->valuesReader = new JsonReader($valuesFile);
        $this->valuesEnvReader = $valuesEnvFile ? new JsonReader($valuesEnvFile) : null;
        $this->provider = $provider;
        $this->env = $env;
    }

    public function getValues(): array
    {
        if (!$this->context) {
            $this->context = $this->postProcess(array_merge(
                $this->valuesReader->readAsJson(),
                $this->valuesEnvReader ? $this->valuesEnvReader->readAsJson() : []
            ));
            if (array_key_exists('_env', $this->context)) {
                throw new \RuntimeException('_env is a reserved value - do not define it manually!');
            }
            $this->context['_env'] = $this->env;
        }
        return $this->context;
    }

    private function postProcess(array $array): array
    {
        foreach ($array as $key => $value) {
            if (is_array($value)) {
                if (array_key_exists('$type', $value) && array_key_exists('arg', $value)) {
                    $externalValueType = $value['$type'];
                    $externalValueArg = $value['arg'];
                    switch ($externalValueType) {
                        case 'secret':
                            $array[$key] = $this->provider->getSecret($externalValueArg);
                            break;
                        default:
                            throw new \RuntimeException("External value with type '{$externalValueType}' is not supported!");
                    }
                } else {
                    $array[$key] = $this->postProcess($value);
                }
            }
        }
        return $array;
    }
}
