<?php

declare(strict_types=1);

namespace HHIT\ConfigGenerator\Command;

use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\ConsoleOutputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class SetSecretsCommand extends AbstractCommand
{
    protected static $defaultName = "set-secrets";

    protected function configure()
    {
        parent::configure();
        $this->addArgument('name', InputArgument::REQUIRED, 'The name of the secret');
        $this->addArgument('file', InputArgument::OPTIONAL, 'A file where to read the secret from or "-" for reading from STDIN');
        $this->addOption('local', 'l', InputOption::VALUE_NONE, 'Updates the local vault.');
        $this->addOption('random', 'r', InputOption::VALUE_OPTIONAL, 'Generates a random value.', false);
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $factory = $this->createFactory($input);
        $vault = $factory->createSodiumVault();
        $localVault = $factory->createDotenvVault();

        $errOutput = $output instanceof ConsoleOutputInterface ? $output->getErrorOutput() : $output;
        $io = new SymfonyStyle($input, $errOutput);
        $name = $input->getArgument('name');
        $vaultToUse = $input->getOption('local') ? $localVault : $vault;

        if (null === $vaultToUse) {
            $io->error('The local vault is disabled.');

            return 1;
        }

        if ($localVault === $vaultToUse && !\array_key_exists($name, $vault->list())) {
            $io->error(sprintf('Secret "%s" does not exist in the vault, you cannot override it locally.', $name));

            return 1;
        }

        if (0 < $random = $input->getOption('random') ?? 16) {
            $value = strtr(substr(base64_encode(random_bytes(intval($random))), 0, intval($random)), '+/', '-_');
        } elseif (!$file = $input->getArgument('file')) {
            $value = $io->askHidden('Please type the secret value');

            if (null === $value) {
                $io->warning('No value provided: using empty string');
                $value = '';
            }
        } elseif ('-' === $file) {
            $value = file_get_contents('php://stdin');
        } elseif (is_file($file) && is_readable($file)) {
            $value = file_get_contents($file);
        } elseif (!is_file($file)) {
            throw new \InvalidArgumentException(sprintf('File not found: "%s".', $file));
        } elseif (!is_readable($file)) {
            throw new \InvalidArgumentException(sprintf('File is not readable: "%s".', $file));
        }

        if ($vaultToUse->generateKeys()) {
            $io->success($vaultToUse->getLastMessage());

            if ($vaultToUse === $vault) {
                $io->caution('DO NOT COMMIT THE DECRYPTION KEY FOR THE PROD ENVIRONMENT⚠️');
            }
        }

        $vaultToUse->seal($name, $value);

        $io->success($vaultToUse->getLastMessage() ?? 'Secret was successfully stored in the vault.');

        if (0 < $random) {
            $errOutput->write(' // The generated random value is: <comment>');
            $output->write($value);
            $errOutput->writeln('</comment>');
            $io->newLine();
        }

        if ($vaultToUse === $vault && null !== $localVault->reveal($name)) {
            $io->comment('Note that this secret is overridden in the local vault.');
        }

        return 0;
    }
}
