<?php
declare(strict_types=1);

namespace HHIT\ConfigGenerator\Command;

use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\ConsoleOutputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class GenerateKeysCommand extends AbstractCommand
{
    protected static $defaultName = 'generate-keys';

    protected function configure()
    {
        parent::configure();
        $this->addOption('local', 'l', InputOption::VALUE_NONE, 'Updates the local vault.');
        $this->addOption('rotate', 'r', InputOption::VALUE_NONE, 'Re-encrypts existing secrets with the newly generated keys.');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $factory = $this->createFactory($input);
        $vault = $factory->createSodiumVault();
        $localVault = $factory->createDotenvVault();

        $io = new SymfonyStyle($input, $output instanceof ConsoleOutputInterface ? $output->getErrorOutput() : $output);
        $vaultToUse = $input->getOption('local') ? $localVault : $vault;

        if (null === $vaultToUse) {
            $io->success('The local vault is disabled.');

            return 1;
        }

        if (!$input->getOption('rotate')) {
            if ($vaultToUse->generateKeys()) {
                $io->success($vaultToUse->getLastMessage());

                if ($vault === $vaultToUse) {
                    $io->caution('DO NOT COMMIT THE DECRYPTION KEY FOR THE PROD ENVIRONMENT⚠️');
                }

                return 0;
            }

            $io->warning($vaultToUse->getLastMessage());

            return 1;
        }

        $secrets = [];
        foreach ($vaultToUse->list(true) as $name => $value) {
            if (null === $value) {
                $io->error($vaultToUse->getLastMessage());

                return 1;
            }

            $secrets[$name] = $value;
        }

        if (!$vaultToUse->generateKeys(true)) {
            $io->warning($vaultToUse->getLastMessage());

            return 1;
        }

        $io->success($vaultToUse->getLastMessage());

        if ($secrets) {
            foreach ($secrets as $name => $value) {
                $vaultToUse->seal($name, $value);
            }

            $io->comment('Existing secrets have been rotated to the new keys.');
        }

        if ($vault === $vaultToUse) {
            $io->caution('DO NOT COMMIT THE DECRYPTION KEY FOR THE PROD ENVIRONMENT⚠️');
        }

        return 0;
    }
}
