<?php
declare(strict_types=1);

namespace HHIT\ConfigGenerator\Generator;

use HHIT\ConfigGenerator\Generator\Definition\DefinitionReader;
use HHIT\ConfigGenerator\Generator\Secrets\SecretProviderFactory;
use HHIT\ConfigGenerator\Generator\Secrets\SymfonyVaultSecretProvider;
use HHIT\ConfigGenerator\Generator\Validator\ValidatorFactory;
use HHIT\ConfigGenerator\Generator\Values\ValuesLoaderFactory;
use Symfony\Bundle\FrameworkBundle\Secrets\DotenvVault;
use Symfony\Bundle\FrameworkBundle\Secrets\SodiumVault;
use Symfony\Component\Dotenv\Dotenv;

class Factory
{

    private string $projectDir;
    private string $env;

    public function __construct(string $projectDir, string $env = 'dev')
    {
        $this->projectDir = $projectDir;
        $this->env = $env;
    }

    private function createDefinitionReader(): DefinitionReader
    {
        return new DefinitionReader($this->projectDir);
    }

    function createSodiumVault(): SodiumVault
    {
        return $this->createSodiumVaultInternal($this->projectDir . '/config/secrets/' . $this->env);
    }

    private function createSodiumVaultInternal(string $secretsDir, $decryptionKey = null): SodiumVault
    {
        return new SodiumVault($secretsDir, $decryptionKey);
    }

    private function createDotenvVault(string $dotenvFile): DotenvVault
    {
        return new DotenvVault($dotenvFile);
    }

    private function createSymfonyVaultSecretProvider(): SymfonyVaultSecretProvider
    {
        return new SymfonyVaultSecretProvider(
            $this->createSodiumVault(),
            $this->createDotenvVault($this->projectDir . '/.env')
        );
    }

    private function createSecretProviderFactory(): SecretProviderFactory
    {
        return new SecretProviderFactory($this->createSymfonyVaultSecretProvider());
    }

    private function createValuesLoaderFactory(): ValuesLoaderFactory
    {
        return new ValuesLoaderFactory($this->createSecretProviderFactory());
    }

    private function createValidatorFactory(): ValidatorFactory
    {
        return new ValidatorFactory();
    }

    function createGenerator(): Generator
    {
        return new Generator(
            $this->createDefinitionReader(),
            $this->createValuesLoaderFactory(),
            $this->createValidatorFactory(),
        );
    }

    function bootEnv()
    {
        if (is_array($env = @include $this->projectDir . '/.env.local.php') && (!isset($env['APP_ENV']) || ($_SERVER['APP_ENV'] ?? $_ENV['APP_ENV'] ?? $env['APP_ENV']) === $env['APP_ENV'])) {
            (new Dotenv(false))->populate($env);
        } else {
            $dotenvFile = $this->projectDir . '/.env';
            if (file_exists($dotenvFile)) {
                (new Dotenv(false))->loadEnv($dotenvFile);
            } else {
                // TODO: log?
            }
        }
    }
}
