<?php
declare(strict_types=1);

namespace HHIT\ConfigGenerator\Generator;

use HHIT\ConfigGenerator\Generator\Compiler\Compiler;
use HHIT\ConfigGenerator\Generator\Definition\DefinitionReader;
use HHIT\ConfigGenerator\Generator\IO\FileWriter;
use HHIT\ConfigGenerator\Generator\Definition\Definition;
use HHIT\ConfigGenerator\Generator\Validator\ValidationException;
use HHIT\ConfigGenerator\Generator\Validator\ValidatorFactory;
use HHIT\ConfigGenerator\Generator\Values\ValuesLoaderFactory;
use Symfony\Component\Console\Output\OutputInterface;

class Generator
{
    /**
     * @var DefinitionReader
     */
    private $definitionReader;
    /**
     * @var ValuesLoaderFactory
     */
    private $valuesLoaderFactory;
    /**
     * @var ValidatorFactory
     */
    private $validatorFactory;

    public function __construct(
        DefinitionReader $definitionReader,
        ValuesLoaderFactory $valuesLoaderFactory,
        ValidatorFactory $validatorFactory
    ) {
        $this->definitionReader = $definitionReader;
        $this->valuesLoaderFactory = $valuesLoaderFactory;
        $this->validatorFactory = $validatorFactory;
    }

    public function processConfigurations(
        string $vaultType,
        bool $overwrite,
        ?string $configurationFile = null,
        ?OutputInterface $output = null
    ): bool {
        $success = [];
        foreach ($this->definitionReader->read($configurationFile) as $definition) {
            /**
             * @var $definition Definition
             */
            $this->writeln($output, "<info>- processing {$definition->getId()}</info>");

            try {
                $valuesLoader = $this->valuesLoaderFactory->create($definition->getValuesFile(), $vaultType);
                $compiler = new Compiler($definition->getTemplateFile(), $valuesLoader);
                $content = $compiler->compile();
                try {
                    $this->validatorFactory->createValidator($definition->getType())->validate($content);
                    $definition->getDestinationFile()->parent(false);
                    $writer = new FileWriter($definition->getDestinationFile());
                    $writer->write($compiler->compile(), $overwrite);
                    $this->writeln($output, "<info>-- configuration written to {$definition->getDestinationFile()->asString()}</info>");
                    $success[] = true;
                } catch (ValidationException $ve) {
                    $this->writeln($output, "<error>== generated content for {$definition->getId()} is invalid: {$ve->getMessage()}</error>");
                    throw $ve;
                }
            } catch (\Throwable $t) {
                if (!$t instanceof ValidationException) {
                    $this->writeln($output, "<error>== generating configuration file {$definition->getId()} failed: {$t->getMessage()}</error>");
                }
                $success[] = false;
            }
        }

        return empty($success) ? true : !in_array(false, $success);
    }

    private function writeln(?OutputInterface $output, $message)
    {
        if ($output) {
            $output->writeln($message);
        }
    }
}
