<?php
declare(strict_types=1);

namespace HHIT\ConfigGenerator\Command;

use Symfony\Component\Console\Helper\Dumper;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\ConsoleOutputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class ListSecretsCommand extends AbstractCommand
{
    protected static $defaultName = "list-secrets";

    protected function configure()
    {
        parent::configure();
        $this->addOption('reveal', 'r', InputOption::VALUE_NONE, 'Display decrypted values alongside names');
    }

    public function execute(InputInterface $input, OutputInterface $output)
    {
        $factory = $this->createFactory($input);
        $vault = $factory->createSodiumVault();
        $localVault = $factory->createDotenvVault();

        $io = new SymfonyStyle($input, $output instanceof ConsoleOutputInterface ? $output->getErrorOutput() : $output);

        if (!$reveal = $input->getOption('reveal')) {
            $io->comment(sprintf('To reveal the secrets run <info>php %s %s --reveal</info>', $_SERVER['PHP_SELF'], $this->getName()));
        }

        $secrets = $vault->list($reveal);
        $localSecrets = null !== $localVault ? $localVault->list($reveal) : null;

        $rows = [];

        $dump = new Dumper($output);
        $dump = static function (?string $v) use ($dump) {
            return null === $v ? '******' : $dump($v);
        };

        foreach ($secrets as $name => $value) {
            $rows[$name] = [$name, $dump($value)];
        }

        if (null !== $message = $vault->getLastMessage()) {
            $io->comment($message);
        }

        foreach ($localSecrets ?? [] as $name => $value) {
            if (isset($rows[$name])) {
                $rows[$name][] = $dump($value);
            }
        }

        if (null !== $localVault && null !== $message = $localVault->getLastMessage()) {
            $io->comment($message);
        }

        (new SymfonyStyle($input, $output))
            ->table(['Secret', 'Value'] + (null !== $localSecrets ? [2 => 'Local Value'] : []), $rows);

        return 0;
    }
}
