<?php
declare(strict_types=1);

namespace HHIT\ConfigGenerator\Generator;

use HHIT\ConfigGenerator\Generator\Definition\DefinitionReader;
use HHIT\ConfigGenerator\Generator\Secrets\SecretProviderFactory;
use HHIT\ConfigGenerator\Generator\Secrets\SymfonyVaultSecretProvider;
use HHIT\ConfigGenerator\Generator\Validator\ValidatorFactory;
use HHIT\ConfigGenerator\Generator\Values\ValuesLoaderFactory;
use Symfony\Bundle\FrameworkBundle\Secrets\DotenvVault;
use Symfony\Bundle\FrameworkBundle\Secrets\SodiumVault;
use Symfony\Component\Dotenv\Dotenv;
use function HHIT\ConfigGenerator\cfgen_private_key_path;
use function HHIT\ConfigGenerator\cfgen_secrets_directory;

class Factory
{
    /**
     * @var string
     */
    private $projectDir;
    /**
     * @var string
     */
    private $env;

    public function __construct(string $projectDir, string $env = 'dev')
    {
        $this->projectDir = $projectDir;
        $this->env = $env;
    }

    private function createDefinitionReader(): DefinitionReader
    {
        return new DefinitionReader($this->projectDir);
    }

    public function dumpPrivateKey()
    {
        $file = cfgen_private_key_path($this->projectDir, $this->env);
        if (!file_exists($file)) {
            throw new \RuntimeException("Key file {$file} does not exist!");
        }
        $key = include($file);

        return base64_encode(rawurldecode(str_replace('\x', '%', $key)));
    }


    public function savePrivateKey(string $key)
    {
        $file = cfgen_private_key_path($this->projectDir, $this->env);
        $dirname = dirname($file);
        if (file_exists($file)) {
            throw new \RuntimeException("Key file {$file} already exists!");
        }
        if (!file_exists($dirname)) {
            throw new \RuntimeException("Directory {$dirname} does not exist!");
        }

        $data = str_replace('%', '\x', rawurlencode(base64_decode($key)));
        $data = sprintf("<?php // %s on %s\n\nreturn \"%s\";\n", basename($file), date('r'), $data);

        if (false === file_put_contents($file, $data, \LOCK_EX)) {
            $e = error_get_last();
            throw new \ErrorException($e['message'] ?? 'Failed to write secrets data.', 0, $e['type'] ?? \E_USER_WARNING);
        }
    }

    public function createSodiumVault(): SodiumVault
    {
        return $this->createSodiumVaultInternal(cfgen_secrets_directory($this->projectDir, $this->env));
    }

    private function createSodiumVaultInternal(string $secretsDir, $decryptionKey = null): SodiumVault
    {
        return new SodiumVault($secretsDir, $decryptionKey);
    }

    public function createDotenvVault(): DotenvVault
    {
        return $this->createDotenvVaultInternal();
    }

    private function createDotenvVaultInternal(): DotenvVault
    {
        return new DotenvVault($this->projectDir . '/.env.' . $this->env . '.local');
    }

    private function createSymfonyVaultSecretProvider(): SymfonyVaultSecretProvider
    {
        return new SymfonyVaultSecretProvider(
            $this->createSodiumVault(),
            $this->createDotenvVault()
        );
    }

    private function createSecretProviderFactory(): SecretProviderFactory
    {
        return new SecretProviderFactory($this->createSymfonyVaultSecretProvider());
    }

    private function createValuesLoaderFactory(): ValuesLoaderFactory
    {
        return new ValuesLoaderFactory($this->createSecretProviderFactory(), $this->env);
    }

    private function createValidatorFactory(): ValidatorFactory
    {
        return new ValidatorFactory();
    }

    public function createGenerator(): Generator
    {
        return new Generator(
            $this->createDefinitionReader(),
            $this->createValuesLoaderFactory(),
            $this->createValidatorFactory()
        );
    }

    public function bootEnv()
    {
        if (is_array($env = @include $this->projectDir . '/.env.local.php') && (!isset($env['APP_ENV']) || ($_SERVER['APP_ENV'] ?? $_ENV['APP_ENV'] ?? $env['APP_ENV']) === $env['APP_ENV'])) {
            (new Dotenv(false))->populate($env);
        } else {
            $dotenvFile = $this->projectDir . '/.env';
            if (file_exists($dotenvFile)) {
                (new Dotenv(false))->loadEnv($dotenvFile);
            } else {
                // TODO: log?
            }
        }
    }
}
