const express = require('express');
const app = express();
const cors = require('cors');
const cookieParser = require('cookie-parser');
const https = require('https');
const fs = require('fs');
const {argv} = require('yargs');

const sameSite = {
	ui: 'https://site.domain.de',
	auth: 'https://auth.site.domain.de'
};

const crossSite = {
	ui: 'https://test-a.azurewebsites.net',
	auth: 'https://test-b.azurewebsites.net'
};

const isCrossSite = argv['cross-site'];
const site = isCrossSite ? crossSite : sameSite;

const corsMiddleware = cors({
	origin: site.ui,
	credentials: true,
});

app.set('views', './views');
app.set('view engine', 'pug');
app.use(express.static('public'))
app.use(corsMiddleware);
app.use(cookieParser());

app.options('*', corsMiddleware)

const viewModel = {
	site,
	cross: isCrossSite
};

app.get(
	'/',
	(req, res) => {
		res.render('index', viewModel);
	}
);

app.get(
	'/check-cookies',
	(req, res) => {
		res.render('check-cookies', viewModel);
	}
);

app.get(
	'/set-cookies',
	(req, res) => {
		res.cookie('test-lax', 'lax-value', {sameSite: 'lax', secure: true});
		res.cookie('test-strict', 'strict-value', {sameSite: 'strict', secure: true});
		res.cookie('test-empty', 'empty-value', {secure: true});
		res.cookie('test-none', 'none-value', {sameSite: 'none', secure: true});

		res.header('Location', site.ui);
		res.sendStatus(302);
	}
);

app.get(
	'/same-site-cookies',
	(req, res) => {
		const val = ['test-lax', 'test-strict', 'test-empty', 'test-none'].reduce((prev, cur) => {
			prev[cur] = !!req.cookies[cur];
			return prev;
		}, {});
		res.json(val);
	}
);

const sslOptions = {
	key: fs.readFileSync('cert.key'),
	cert: fs.readFileSync('cert.pem')
};

const banner = `
SameSite Testing Application
============================

Add 

 - site.domain.de
 - auth.site.domain.de
 - a-test.azurewebsites.net
 - b-test.azurewebsites.net

to your local hosts file, pointing to localhost.

=> ======================================================
=> Do not forget to remove these entries after testing !=
=> ======================================================

Application is currently listening at

	${site.ui}
	
in

	${isCrossSite ? 'cross-site' : 'same-site'} mode.
`;

https.createServer(sslOptions, app).listen(
	443,
	() => {
		console.log(banner)
	}
);
